# Script: ctx_stf.ps1
# Author: Roland Rusch, easy-smart solution GmbH
# Description: Query Citrix StoreFront Server Information
#
#
# USAGE:
#   as a script:    %SystemRoot%\system32\WindowsPowerShell\v1.0\powershell.exe -NoLogo -NoProfile -ExecutionPolicy Bypass -File "C:\bin\zabbix\citrix\ctx_stf.ps1" "<ITEM_TO_QUERY>" "[ID]"
#   as an item:     ctx_stf[<ITEM_TO_QUERY>,<ID>]
#

$version = "3"
$jobname = "zabbix_ctx_ls"
$logFile = "$env:TEMP\$jobname.log"

$ITEM = [string]$args[0]
$ID = [string]$args[1]



Import-Module Citrix.StoreFront


# Get Citrix StoreFront Version
$serviceImagePath = (Get-Process -Name "Citrix.Storefront").Path
$corePath = Split-Path -Path $serviceImagePath
$file = Get-Item -Path $serviceImagePath
$citrix_stf_version_string = $file.VersionInfo.ProductVersion
$citrix_stf_product_string = $file.VersionInfo.ProductName
$citrix_stf_product_version_string = "$citrix_stf_product_string $citrix_stf_version_string"
$citrix_stf_version = $file.VersionInfo.ProductMajorPart * 100 + $file.VersionInfo.ProductMinorPart



function LogLine([String]$logFile = $(Throw 'LogLine:$logFile unspecified'), [String]$row = $(Throw 'LogLine:$row unspecified')) {
	$logDateTime = Get-Date -Format 'yyyy-MM-dd HH:mm:ss'
	Add-Content -Encoding UTF8 $logFile ($logDateTime + " - " + $ITEM + " - " + $row) 
}


function Get-Data {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ITEM,
		
		[Parameter(Mandatory=$False,Position=2)]
		[string]$ID=''
	)
	
	$returnValue="ZBX_NOTSUPPORTED"
	switch ($ITEM) {
		'Script.Version'
		{
			$returnValue = $version
		}

		'Server.Product'
		{
			$returnValue = $citrix_stf_product_string
		}

		'Server.ProductVersion'
		{
			$returnValue = $citrix_stf_version_string
		}

		'Server.FullProductVersion'
		{
			$returnValue = $citrix_stf_product_version_string
		}

		'Server.Version'
		{
			$returnValue = $citrix_stf_version
		}
		
		
		'Store.discover' {
			# Open JSON object
			$output =  "{`"data`":["
			
			$query = Get-STFStoreService
		
			$Index = 0
			$count = ($query | Measure-Object).count
			foreach ($object in $query) {
				$Id = [string]$object.PLD
				
				$output = $output + "{"
				
				$output = $output + "`"{#ID}`":`"$($object.SiteId)`","
				$output = $output + "`"{#FRIENDLYNAME}`":`"$($object.FriendlyName)`","
				$output = $output + "`"{#DATAINDEX}`":`"$Index`""

				$output = $output + "}"

				if ($count -gt 1) {
					$output = $output + ","
				}
				$count--
				$Index++
			}
			# Close JSON object
			$output = $output + "]}"
			$returnValue=$output
		}


		'Store.FriendlyName'
		{
			$store = Get-STFStoreService -SiteId $ID
			$returnValue = $store.Friendlyname
		}


		'Store.HostbaseUrl'
		{
			$store = Get-STFStoreService -SiteId $ID
			$returnValue = "$($store.Routing.HostbaseUrl)"
		}


		'Store.LocalUrl'
		{
			$store = Get-STFStoreService -SiteId $ID
			$webBinding = $store.WebApplication.WebSite.Bindings | Where-Object {$_.HostSource -eq 'MachineName'}
			$returnValue = "$($webBinding.BaseUrl)$($store.VirtualPath)"
		}


		
		default {
			$returnValue="ZBX_NOTSUPPORTED '$ITEM' '$ID'"
		}
	}
	return [string]$returnValue
}



# Query
switch ($ITEM) {

	"Tempdir" {
		Write-Host "$env:TEMP"
	}

	default {
		#write-host "'$ITEM'"
		Get-Data $ITEM $ID
	}
}


