# Script: 3ware.ps1
# Author: Roland Rusch, easy-smart solution GmbH
# Description: Query 3ware RAID information
#
#
# USAGE:
#   as a script:    %SystemRoot%\system32\WindowsPowerShell\v1.0\powershell.exe -NoLogo -NoProfile -ExecutionPolicy Bypass -File "C:\bin\zabbix\3ware_raid\3ware.ps1" "<ITEM_TO_QUERY>" "ARG1"
#   as an item:     3ware[<ITEM_TO_QUERY>,<ARG1>]
#

$version = "1.0.1"

$TW_CLI = "C:\Program Files\3ware\CLI\tw_cli.EXE"

$ITEM = [string]$args[0]
$ARG1 = [string]$args[1]




function Call-Cli {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ARG
	)
	$return=& $TW_CLI $ARG
	return $return.Trim()
}


function Get-ControllerDetail {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$Ctl
	)
	$str=Call-Cli "$Ctl show all" `
		| Select-String -Pattern '^/c\d+\s+'
	$ret=New-Object -TypeName psobject
	foreach($line in $($str -split "`n")) {
		$match=$line.Trim() | Select-String -Pattern '^/c\d+\s+(.*?)\s*=\s*(.*?)\s*$'
		$name=($match.Matches[0].Groups[1].Value).Replace(" ", "")
		$value=($match.Matches[0].Groups[2].Value)
		#Write-Host $name, "=", $value
		$ret | Add-Member -Name $name -Value $value -MemberType Noteproperty
	}
	return $ret
}


function Get-Controller {
	Param(
		[Parameter(Mandatory=$False,Position=1)]
		[string]$Ctl=""
	)
	$ret=Call-Cli "show" `
		| Select-String -Pattern '^c\d+\s+' `
		| ConvertFrom-String -PropertyNames Ctl, Model, Ports, Drives, Units, NotOpt, RRate, VRate, BBU
	foreach($r2 in $ret) {
		$r2.Ctl="/" + $r2.Ctl
	}
	if($Ctl) { return ($ret | Where-Object {$_.Ctl -eq $Ctl}) }
	return $ret
}


function Get-Unit {
	Param(
		[Parameter(Mandatory=$False,Position=1)]
		[string]$Unit=""
	)
	$controllers=Get-Controller
	$ret=@()
	foreach($controller in $controllers) {
		$myCtl=$controller.Ctl
		$r=Call-Cli "$myCtl show" `
			| Select-String -Pattern '^u\d+\s+' `
			| ConvertFrom-String -PropertyNames Unit, UnitType, Status, RCmpl, VIM, Stripe, Size, Cache, AVrfy
		foreach($r2 in $r) {
			$r2.Unit="$myCtl/" + $r2.Unit
			try {
				$r2 | Add-Member -Name PVIM -Value 0 -MemberType Noteproperty
				$r2.PVIM=($r2.VIM | Select-String -Pattern '^(\d+)').Matches[0].Value
			} catch { $r2.PVIM=100 }
			try {
				$r2 | Add-Member -Name PRCmpl -Value 0 -MemberType Noteproperty
				$r2.PRCmpl=($r2.RCmpl | Select-String -Pattern '^(\d+)').Matches[0].Value
			} catch { $r2.PRCmpl=100 }
			$r2.Size=$r2.Size*1024*1024*1024
			$r2 | Add-Member -Name Ctl -Value $myCtl -MemberType Noteproperty
			$ret+=$r2
		}
	}
	if($Unit) { return ($ret | Where-Object {$_.Unit -eq $Unit}) }
	return $ret
}


function Get-Port {
	Param(
		[Parameter(Mandatory=$False,Position=1)]
		[string]$VPort=""
	)
	$controllers=Get-Controller
	$ret=@()
	foreach($controller in $controllers) {
		$myCtl=$controller.Ctl
		$r=Call-Cli "$myCtl show" `
			| Select-String -Pattern '^p\d+\s+'
		foreach($line in $($r -split "`n"))
		{
			#p0    OK             u0   3.63 TB   SATA  0   -            HGST HUS726T4TALA6L4
			$r2=$line | ConvertFrom-String -PropertyNames VPort, Status, Unit, Size, SizeUnit, Type, Phy, EnclSlot, Model
			$r2.VPort="$myCtl/" + $r2.VPort
			$r2.Model=$line.substring(59)
			try { $r2.Size=[float]($line.substring(26) | Select-String -Pattern '^[0-9\.]+').Matches[0].Value }
			catch { $r2.Size=0 }
			if($r2.SizeUnit -eq "MB") {$r2.Size=$r2.Size*1024*1024}
			if($r2.SizeUnit -eq "GB") {$r2.Size=$r2.Size*1024*1024*1024}
			if($r2.SizeUnit -eq "TB") {$r2.Size=$r2.Size*1024*1024*1024*1024}
			$r2 | Add-Member -Name Ctl -Value $myCtl -MemberType Noteproperty
			$ret+=$r2
		}
	}
	if($VPort) { return ($ret | Where-Object {$_.VPort -eq $VPort}) }
	return $ret
}





# Query Information
switch($ITEM) {
	
	
	"version" {
		$return=Call-Cli "show ver" | ConvertFrom-String -Delimiter "="
		$o=@{}
		foreach($obj in $return) {
			#$o=@{$obj.P1.Trim()=$obj.P2.Trim()}
			$o | Add-Member -Name $obj.P1.Trim() -Value $obj.P2.Trim() -MemberType Noteproperty
		}
		Write-Host ($o |ConvertTo-Json)
	}
	
	
	"alarm" {
		$units=Get-Unit $ARG1 | Where-Object {$_.Status -ne "OK" -and $_.Status -ne "VERIFYING"} | Measure
		$ports=Get-Port $ARG1 | Where-Object {$_.Status -ne "OK" -and $_.Status -ne "VERIFYING"} | Measure
		$alarm=0 + $units.count + $ports.count
		Write-Host $alarm
	}
	
	
	"mediaerrors" {
		$ports=Get-Port $ARG1 | Where-Object {$_.Status -ne "OK" -and $_.Status -ne "VERIFYING"} | Measure
		$alarm=0 + $ports.count
		Write-Host $alarm
	}
	
	
	"raiderrors" {
		$units=Get-Unit $ARG1 | Where-Object {$_.Status -ne "OK" -and $_.Status -ne "VERIFYING"} | Measure
		$alarm=0 + $units.count
		Write-Host $alarm
	}
	
	
	
	"controller_discovery" {
		$controllers=Get-Controller $ARG1
		$count = ($controllers | Measure-Object).count
		$output =  "{`"data`":["
		foreach($controller in $controllers) {
			$output+= "{"
			$output+= "`"{#CTL}`":`"$($controller.Ctl)`","
			$output+= "`"{#MODEL}`":`"$($controller.Model)`","
			$output+= "`"{#PORTS}`":`"$($controller.Ports)`","
			$output+= "`"{#DRIVES}`":`"$($controller.Drives)`","
			$output+= "`"{#UNITS}`":`"$($controller.Units)`""
			$output+= "}"
			if ($count -gt 1) { $output+= "," }
			$count--
		}
		$output = $output + "]}"
		Write-Host $output
	}
	
	
	"controller_info" {
		$controller=Get-Controller $ARG1
		$controllerdetail=Get-ControllerDetail $ARG1
		foreach ($Property in ($controller | Get-Member -MemberType Properties)) {
			$arguments += @{$Property.Name = $controller.($Property.name) }
		}
		foreach ($Property in ($controllerdetail | Get-Member -MemberType Properties)) {
			try { $arguments += @{$Property.Name = $controllerdetail.($Property.name) } }
			catch {}
		}
		Write-Host ($arguments |ConvertTo-Json)
	}
	
	
	
	"unit_discovery" {
		$units=Get-Unit $ARG1
		$count = ($units | Measure-Object).count
		$output =  "{`"data`":["
		foreach($unit in $units) {
			$output+= "{"
			$output+= "`"{#UNIT}`":`"$($unit.Unit)`","
			$output+= "`"{#CTL}`":`"$($unit.Ctl)`","
			$output+= "`"{#UNITTYPE}`":`"$($unit.UnitType)`","
			$output+= "`"{#STATUS}`":`"$($unit.Status)`","
			$output+= "`"{#PRCMPL}`":`"$($unit.RPCmpl)`""
			$output+= "}"
			if ($count -gt 1) { $output+= "," }
			$count--
		}
		$output = $output + "]}"
		Write-Host $output
	}
	
	
	"unit_info" {
		$unit=Get-Unit $ARG1
		Write-Host ($unit |ConvertTo-Json)
	}
	
	
	
	"port_discovery" {
		$ports=Get-Port $ARG1
		$count = ($ports | Measure-Object).count
		$output =  "{`"data`":["
		foreach($port in $ports) {
			$output+= "{"
			$output+= "`"{#VPORT}`":`"$($port.VPort)`","
			$output+= "`"{#CTL}`":`"$($port.Ctl)`","
			$output+= "`"{#UNIT}`":`"$($port.Unit)`","
			$output+= "`"{#TYPE}`":`"$($port.Type)`","
			$output+= "`"{#STATUS}`":`"$($port.Status)`","
			$output+= "`"{#MODEL}`":`"$($port.Model)`""
			$output+= "}"
			if ($count -gt 1) { $output+= "," }
			$count--
		}
		$output = $output + "]}"
		Write-Host $output
	}
	
	
	"port_info" {
		$port=Get-Port $ARG1
		Write-Host ($port |ConvertTo-Json)
	}
	
	
}




