# Script: check_windows_updates.ps1
# Author: Roland Rusch, easy-smart solution GmbH
# Description: Search for windows updates and pending reboot
#
#
# USAGE:
#   as a script:    %SystemRoot%\system32\WindowsPowerShell\v1.0\powershell.exe -NoLogo -NoProfile -ExecutionPolicy Bypass -File "C:\bin\zabbix\windowsupdates\check_windows_updates.ps1"
#   as an item:     windowsupdates
#

$htReplace = New-Object hashtable
foreach ($letter in (Write-Output  ae  oe  ue  Ae  Oe  Ue  ss)) {
	$foreach.MoveNext() | Out-Null
	$htReplace.$letter = $foreach.Current
}
$pattern = "[$(-join $htReplace.Keys)]"
$jobname = "zabbix_check_windows_updates"

$returnStateOK = 0
$returnStateWarning = 1
$returnStateCritical = 2
$returnStateUnknown = 4
$returnStatePendingReboot = 3
$returnStateOptionalUpdates = 1

$updateCacheFile = "$env:TEMP\$jobname-cache.xml"
$updateCacheExpireHours = "24"

$logFile = "$env:TEMP\$jobname.log"

function LogLine([String]$logFile = $(Throw 'LogLine:$logFile unspecified'), [String]$row = $(Throw 'LogLine:$row unspecified')) {
	$logDateTime = Get-Date -Format 'yyyy-MM-dd HH:mm:ss'
	Add-Content -Encoding UTF8 $logFile ($logDateTime + " - " + $row) 
}

Try {

	if (Test-Path "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\WindowsUpdate\Auto Update\RebootRequired") {
		#Write-Host "updates installed, reboot required"
		if (Test-Path $logFile) {
			Remove-Item $logFile | Out-Null
		}
		if (Test-Path $updateCacheFile) {
			Remove-Item $updateCacheFile | Out-Null
		}
		Write-Host $returnStatePendingReboot
		exit $returnStatePendingReboot
	}


	# Read cache file, if it exists
	# and check expiration
	if (Test-Path $updateCacheFile) {
		LogLine -logFile $logFile -row ("using valid cache file ($updateCacheFile)...")
		$updates = Import-Clixml $updateCacheFile
	
		# delete cache file, if older than $updateCacheExpireHours
		if ((Get-Date) -gt ((Get-Item $updateCacheFile).LastWriteTime.AddHours($updateCacheExpireHours))) {
			LogLine -logFile $logFile -row ("update cache expired, removing file")
			Remove-Item $updateCacheFile
		}
	}

# Create new cache file if it does not exist
if (-not (Test-Path $updateCacheFile)) {
	LogLine -logFile $logFile -row ("$updateCacheFile not found")
	if( (Get-Job |Where-Object {$_.Name -like $jobname -and $_.State -like "Running"}).count -gt 0) {
		LogLine -logFile $logFile -row ("job already running")
	} else {
		LogLine -logFile $logFile -row ("creating....")
		$job=Start-Job -Name $jobname -ArgumentList @($updateCacheFile) -ScriptBlock {
			$updateSession = new-object -com "Microsoft.Update.Session"
			$updates=$updateSession.CreateupdateSearcher().Search(("IsInstalled=0 and Type='Software'")).Updates
			Export-Clixml -InputObject $updates -Encoding UTF8 -Path $args[0]
		}
	}
	#LogLine -logFile $logFile -row (ConvertTo-Json -InputObject $job)
	LogLine -logFile $logFile -row ("Job status: " + $job.State)
	
	# exit with status "unknown" if no data was read
	if( -not $updates ) {
		Write-Host $returnStateUnknown
		exit $returnStateUnknown
	}
}




$critialTitles = "";
$countCritical = 0;
$countOptional = 0;
$countHidden = 0;

if ($updates.Count -eq 0) {
	#Write-Host "OK - no pending updates.|critical=$countCritical;optional=$countOptional;hidden=$countHidden"
	Write-Host $returnStateOK
	exit $returnStateOK
}

foreach ($update in $updates) {
	if ($update.IsHidden) {
		$countHidden++
	}
	elseif ($update.AutoSelectOnWebSites) {
		$criticalTitles += $update.Title + " "
		$countCritical++
	} else {
		$countOptional++
	}
}
if (($countCritical + $countOptional) -gt 0) {
	$returnString = "Updates: $countCritical critical, $countOptional optional" + [Environment]::NewLine + "$criticalTitles"
	$returnString = [regex]::Replace($returnString, $pattern, { $htReplace[$args[0].value] })
	
	# 1024 chars max, reserving 48 chars for performance data -> 
	if ($returnString.length -gt 976) {
		#Write-Host ($returnString.SubString(0,975) + "|critical=$countCritical;optional=$countOptional;hidden=$countHidden")
	} else {
		#Write-Host ($returnString + "|critical=$countCritical;optional=$countOptional;hidden=$countHidden")
	}
}

#if ($countCritical -gt 0 -or $countOptional -gt 0) {
#	Start-Process "wuauclt.exe" -ArgumentList "/detectnow" -WindowStyle Hidden
#}

if ($countCritical -gt 0) {
	Write-Host $returnStateCritical
	exit $returnStateCritical
}

if ($countOptional -gt 0) {
	Write-Host $returnStateOptionalUpdates
	exit $returnStateOptionalUpdates
}

if ($countHidden -gt 0) {
	#Write-Host "OK - $countHidden hidden updates.|critical=$countCritical;optional=$countOptional;hidden=$countHidden"
	Write-Host $returnStateOK
	exit $returnStateOK
}


}
catch {
	LogLine -logFile $logFile -row ($_)
	LogLine -logFile $logFile -row ($_.InvocationInfo.PositionMessage)
}


Write-Host $returnStateUnknown
exit $returnStateUnknown

