# Script: vbrjob.ps1
# Author: Roland Rusch, easy-smart solution GmbH
# Description: Query Veeam Information
#
#
# USAGE:
#   as a script:    %SystemRoot%\system32\WindowsPowerShell\v1.0\powershell.exe -NoLogo -NoProfile -ExecutionPolicy Bypass -File "C:\bin\zabbix\veeam\vbrjob.ps1" "<ITEM_TO_QUERY>" "[JOBID]"
#   as an item:     vbr[<ITEM_TO_QUERY>,<JOBID>]
#

$version = "1.0.10.6"
$version_num = 1001006
$jobname = "zabbix_vbrjob"
$logFile = "$env:TEMP\$jobname.log"
$cacheExpireMinutes = "300"

$ITEM = [string]$args[0]
$ID = [string]$args[1]


# Get Veeam Version
$corePath = Get-ItemProperty -Path "HKLM:\Software\Veeam\Veeam Backup and Replication\" -Name "CorePath"
$depDLLPath = Join-Path -Path $corePath.CorePath -ChildPath "Packages\VeeamDeploymentDll.dll" -Resolve
$file = Get-Item -Path $depDLLPath
$veeam_version_string = $file.VersionInfo.ProductVersion
$veeam_product_string = $file.VersionInfo.ProductName
$veeam_product_version_string = "$veeam_product_string v$veeam_version_string"
$veeam_version = $file.VersionInfo.ProductMajorPart * 100 + $file.VersionInfo.ProductMinorPart

# Load Veeam module/snapin depending on availability
if (-Not (Get-Module -ListAvailable -Name Veeam.Backup.PowerShell) ) {
	Add-PSSnapin -PassThru VeeamPSSnapIn -ErrorAction Stop | Out-Null
} else {
	Import-Module "Veeam.Backup.PowerShell" -ErrorAction SilentlyContinue -WarningAction SilentlyContinue
}

# Open connection to Veeam server
if( $veeam_version -ge 905 ) {
	Connect-VBRServer -ErrorAction SilentlyContinue
}



$localServer=$null
$backupSession=$null
$runningJobs=$null
$job=$null
$session=$null
$backup=$null
$repo=$null




function LogLine([String]$logFile = $(Throw 'LogLine:$logFile unspecified'), [String]$row = $(Throw 'LogLine:$row unspecified')) {
	$logDateTime = Get-Date -Format 'yyyy-MM-dd HH:mm:ss'
	Add-Content -Encoding UTF8 $logFile ($logDateTime + " - " + $ITEM + " - " + $row) 
}


function Get-Cache {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ID,
		[Parameter(Mandatory=$False,Position=2)]
		[string]$type='generic'
	)
	
	$cacheFilename = "$env:TEMP\$jobname-$type-$ID-cache.xml"
	# Read cache file, if it exists
	# and check expiration
	$data=$False
	
	if (Test-Path $cacheFilename) {
		# delete cache file, if older than $cacheExpireHours
		if ((Get-Date) -gt ((Get-Item $cacheFilename).LastWriteTime.AddMinutes($cacheExpireMinutes))) {
			LogLine -logFile $logFile -row ("cache file expired, removing ($cacheFilename)")
			Remove-Item $cacheFilename
		}
	}
	
	if (Test-Path $cacheFilename) {
		LogLine -logFile $logFile -row ("reading cache file ($cacheFilename)...")
		$data = Import-Clixml $cacheFilename
	}
	return $data
}


function Set-Cache {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[object]$data,
		[Parameter(Mandatory=$True,Position=2)]
		[string]$ID,
		[Parameter(Mandatory=$False,Position=3)]
		[string]$type='generic'
	)
	$cacheFilename = "$env:TEMP\$jobname-$type-$ID-cache.xml"
	
	if( Test-Path $cacheFilename ) {
			# delete cache file, if older than $cacheExpireHours
		if ((Get-Date) -gt ((Get-Item $cacheFilename).LastWriteTime.AddMinutes($cacheExpireMinutes))) {
			LogLine -logFile $logFile -row ("cache file expired, removing ($cacheFilename)")
			Remove-Item $cacheFilename
		}
	}
	
	# Create new cache file if it does not exist
	if (-not (Test-Path $cacheFilename)) {
		LogLine -logFile $logFile -row ("cache file not found, creating new ($cacheFilename)")
		Export-Clixml -InputObject $data -Encoding UTF8 -Path $cacheFilename
	}
}


function Get-Job {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ID
	)
	
	if( !$global:job ) { $global:job=@{} }
	if( !$global:job[$ID] ) {
		$global:job[$ID] = Get-VBRJob | Where-Object {$_.Id -like "*$ID*"}
	}
	return $global:job[$ID]
}


function Get-Session {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ID
	)
	
	if( !$global:session ) { $global:session=@{} }
	if( !$global:session[$ID] ) {
		$job=Get-Job $ID

		if ($job.JobType -eq "SimpleBackupCopyPolicy") {
			# Letzt Session des Jobs holen
			$global:session[$ID] = $job.GetWorkerJobs().FindLastSession()
		} else {
			# Letzte abgeschlossene Session des Jobs holen
			$global:session[$ID] = (Get-VBRBackupSession | Where-Object {$_.JobId -eq $job.Id} | Sort-Object Endtime -Descending)[0]
		}
	}
	return $global:session[$ID]
}


function Get-Backup {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ID
	)
	
	if( !$global:backup ) { $global:backup=@{} }
	if( !$global:backup[$ID] ) {
		$global:backup[$ID] = Get-VBRBackup | Where-Object {$_.JobId -like "*$ID*"}
	}
	return $global:backup[$ID]
}


function Get-Repository {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ID
	)
	
	if( !$global:repo ) { $global:repo=@{} }
	if( !$global:repo[$ID] ) {
		$global:repo[$ID] = Get-VBRBackupRepository | Where-Object {$_.Id -like "*$ID*"}
	}
	return $global:repo[$ID]
}


function Get-JobObject {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ID
	)
	
	$object=Get-Cache $ID 'Job.Json'
	if( !$object ) {
		$object=@{
			JobId			= Get-VeeamData 'JobId' $ID;
			Name			= Get-VeeamData 'Name' $ID;
			Status			= Get-VeeamData 'Status' $ID;
			LastMessage		= Get-VeeamData 'LastMessage' $ID;
			Scheduled		= Get-VeeamData 'Scheduled' $ID;
			RunStatus		= Get-VeeamData 'RunStatus' $ID;
			IncludedSize	= Get-VeeamData 'IncludedSize' $ID;
			ExcludedSize	= Get-VeeamData 'ExcludedSize' $ID;
			VmCount			= Get-VeeamData 'VmCount' $ID;
			Type			= Get-VeeamData 'Type' $ID
		}
		Set-Cache $object $ID 'Job.Json'
	}
	return $object
}


function Get-RepoObject {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ID
	)
	
	$object=Get-Cache $ID 'Repo.Json'
	if( !$object ) {
		$object=@{
			RepoId			= Get-VeeamData 'RepoId' $ID;
			Name			= Get-VeeamData 'RepoName' $ID;
			Path			= Get-VeeamData 'RepoPath' $ID;
			TotalSpace		= Get-VeeamData 'RepoTotalSpace' $ID;
			FreeSpace		= Get-VeeamData 'RepoFreeSpace' $ID;
			Type			= Get-VeeamData 'RepoType' $ID
		}
		Set-Cache $object $ID 'Repo.Json'
	}
	return $object
}


function Get-VeeamData {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ITEM,
		
		[Parameter(Mandatory=$False,Position=2)]
		[string]$ID=''
	)
	
	$returnValue="ZBX_NOTSUPPORTED"
	switch ($ITEM) {
		'Version'
		{
			$returnValue = $version_num
		}

		'Server.Product'
		{
			$returnValue = $veeam_product_string
		}

		'Server.ProductVersion'
		{
			$returnValue = $veeam_version_string
		}

		'Server.FullProductVersion'
		{
			$returnValue = $veeam_product_version_string		}

		'Server.Version'
		{
			$returnValue = $veeam_version
		}
		
		'Server.Name' {
			$global:localServer=if( !$global:localServer )
				{ Get-VBRServer -Type Local }
			$returnValue=$localServer.Name
		}
		
		'WorkingJob' {
			$global:backupSession=if( !$global:backupSession )
				{ Get-VBRBackupSession | Where-Object {$_.IsWorking -eq $true } | Measure-Object }
			if($backupSession) {
				$returnValue=$backupSession.Count
			} else {
				$returnValue="0"
			}
		}
		
		'RunningJob' {
			$global:runningJobs=if( !$global:runningJobs )
				{ Get-VBRJob | Where-Object {$_.IsRunning -eq $true} }
			if($runningJobs) {
				$returnValue=($runningJobs |Measure-Object).Count
			} else {
				$returnValue="0"
			}
		}
		
		'RunningBackupJob' {
			$global:runningJobs=if( !$global:runningJobs )
				{ Get-VBRJob | Where-Object {$_.IsRunning -eq $true} }
			$query = $runningJobs | Where-Object {$_.IsRunning -eq $true -and $_.IsBackup -eq $true} |Measure-Object
			if ($query) {
				$returnValue=($query |Measure-Object).Count
			} else {
				$returnValue="0"
			}
		}
		
		
		
		'discover_repos' {
			# Open JSON object
			$output =  "{`"data`":["
			$query = Get-VBRBackupRepository | Select-Object Id,Name,Type
			$count = ($query | Measure-Object).count
			foreach ($object in $query) {
				$Id = [string]$object.Id
				$Name = [string]$object.Name
				$Type = [string]$object.Type
				if ($count -eq 1) {
					$output = $output + "{`"{#REPOID}`":`"$Id`",`"{#REPONAME}`":`"$Name`",`"{#REPOTYPE}`":`"$Type`"}"
				} else {
					$output = $output + "{`"{#REPOID}`":`"$Id`",`"{#REPONAME}`":`"$Name`",`"{#REPOTYPE}`":`"$Type`"},"
				}
				$count--
			}
			# Close JSON object
			$output = $output + "]}"
			$returnValue=$output
		}
		
		
		'RepoName' {
			$repo = Get-Repository $ID
			$returnValue=$repo.Name
		}
		
		'RepoId' {
			$repo = Get-Repository $ID
			$returnValue=$repo.Id
		}
		
		'RepoPath' {
			$repo = Get-Repository $ID
			$returnValue=$repo.Path
		}
		
		'RepoType' {
			$repo = Get-Repository $ID
			$returnValue=$repo.Type
		}
		
		'RepoTotalSpace' {
			$repo = Get-Repository $ID
			if($veeam_version -lt 1100) {
				$returnValue=$repo.Info.CachedTotalSpace
			} else {
				$returnValue=$repo.GetContainer().CachedTotalSpace.InBytes
			}
		}
		
		'RepoFreeSpace' {
			$repo = Get-Repository $ID
			if($veeam_version -lt 1100) {
				$returnValue=$repo.Info.CachedFreeSpace
			} else {
				$returnValue=$repo.GetContainer().CachedFreeSpace.InBytes
			}
		}
		
		
		
		'Discovery' {
			# Open JSON object
			$output =  "{`"data`":["
			# $query = Get-VBRJob | Where-Object {$_.IsScheduleEnabled -eq "true"} | Select-Object Id,Name, IsScheduleEnabled
			$query = Get-VBRJob | Select-Object Id,Name, IsScheduleEnabled
			$count = ($query | Measure-Object).count
			foreach ($object in $query) {
				$Id = [string]$object.Id
				$Name = [string]$object.Name
				$Schedule = [string]$object.IsScheduleEnabled
				if ($count -eq 1) {
					$output = $output + "{`"{#JOBID}`":`"$Id`",`"{#JOBNAME}`":`"$Name`",`"{#JOBSCHEDULED}`":`"$Schedule`"}"
				} else {
					$output = $output + "{`"{#JOBID}`":`"$Id`",`"{#JOBNAME}`":`"$Name`",`"{#JOBSCHEDULED}`":`"$Schedule`"},"
				}
				$count--
			}
			# Close JSON object
			$output = $output + "]}"
			$returnValue=$output
		}
		
		
		'Status' {
			$job = Get-Job $ID
			if ($job) {
				if($job.IsScheduleEnabled -eq $true) {
					$result=[string]$job.GetLastResult()
					if( ($result -eq "None") -or ($job.JobType -eq "SimpleBackupCopyPolicy") ) {
						$session = Get-Session $ID
						$result=$session.Result
					}
				} else {
					$result="Disabled"
				}
				switch ($result) {
					"Failed" {
						$returnValue="0"
					}
					"Warning" {
						$returnValue="1"
					}
					"Success" {
						$returnValue="2"
					}
					"None" {
						$returnValue="5"
					}
					"Disabled" {
						$returnValue="3"
					}
					default {
						$returnValue="4"
					}
				}
			}
			else { $returnValue="4"}
		}
		
		'LastMessage' {
			$session = Get-Session $ID
			if ($session) {
				$logrecords=$session.Logger.GetLog().UpdatedRecords | Where-Object {$_.Status -ne "ESucceeded"} | Sort-Object StartTime
				if ( ($logrecords | Measure-Object).count -gt 0 ) {
					# Es gibt mehrere Log-Einträge, die nicht ESucceeded sind
					$returnArray=@()
					foreach ($obj in $logrecords) {$returnArray += @(-join($obj.Status, ": ", $obj.Title))}
					$returnValue=[string]($returnArray -join [Environment]::NewLine)
				} else {
					# Keine Log-Einträge gefunden => letzten Eintrag nehmen
					$logrecord=$session.Logger.GetLog().UpdatedRecords[0]
					if($logrecord) {
						$returnValue= -join($logrecord.Status, ": ", $logrecord.Title)
					} else {
						$returnValue="N/A"
					}
				}
			} else { $returnValue = "ERROR: Session not found." }
		}
		
		'Scheduled' {
			$job = Get-Job $ID
			$returnValue = if ($job.IsScheduleEnabled) { "1" } else { "0"}
		}
		
		'RunStatus' {
			$job = Get-Job $ID
			$returnValue = if ($job.IsRunning) { "1" } else { "0"}
		}
		
		'IncludedSize' {
			$job = Get-Job $ID
			$returnValue=$job.Info.IncludedSize
		}
		
		'ExcludedSize' {
			$job = Get-Job $ID
			$returnValue=$job.Info.ExcludedSize
		}
		
		'VmCount' {
			$backup = Get-Backup $ID
			$returnValue=$backup.VmCount
		}
		
		'Type' {
			$job = Get-Job $ID
			$returnValue=$job.JobType
		}
		
		'Name' {
			$job = Get-Job $ID
			$returnValue=$job.Name
		}
		
		'JobId' {
			$job = Get-Job $ID
			$returnValue=$job.Id
		}
		
		
		'warmcache' {
			$cacheExpireMinutes = "0";
			$jobs = Get-VBRJob | Select-Object Id,Name, IsScheduleEnabled
			foreach ($object in $jobs) {
				$Id = [string]$object.Id
				Get-JobObject $Id
			}
			$repos = Get-VBRBackupRepository | Select-Object Id,Name,Type
			foreach ($object in $repos) {
				$Id = [string]$object.Id
				Get-RepoObject $Id
			}
		}
		
		
		default {
			$returnValue="ZBX_NOTSUPPORTED '$ITEM' '$ID'"
		}
	}
	return [string]$returnValue
}



# Query VEEAM
switch ($ITEM) {
	
	"Server.Json" {
		$object=Get-Cache 'NULL' 'Server.Json'
		if( !$object ) {
			$object=@{
				Version=Get-VeeamData 'Version';
				Server_Version=Get-VeeamData 'Server.Version';
				Server_Product=Get-VeeamData 'Server.Product';
				Server_ProductVersion=Get-VeeamData 'Server.ProductVersion';
				Server_FullProductVersion=Get-VeeamData 'Server.FullProductVersion';
				Server_Name=Get-VeeamData 'Server.Name';
				WorkingJob=Get-VeeamData 'WorkingJob';
				RunningJob=Get-VeeamData 'RunningJob';
				RunningBackupJob=Get-VeeamData 'RunningBackupJob'
			}
			Set-Cache $object 'NULL' 'Server.Json'
		}
		Write-Host ($object |ConvertTo-Json)
	}
	
	"Job.Json" {
		$object=Get-JobObject $ID
		Write-Host ($object |ConvertTo-Json)
	}
	
	"Repo.Json" {
		$object=Get-RepoObject $ID
		Write-Host ($object |ConvertTo-Json)
	}
	
	"Tempdir" {
		Write-Host "$env:TEMP"
	}

	default {
		#write-host "'$ITEM'"
		Get-VeeamData $ITEM $ID
	}
}

# Close connection to Veeam server
if( $veeam_version -ge 905 ) {
	Disconnect-VBRServer
}

