# Script: megaraid.ps1
# Author: Roland Rusch, easy-smart solution GmbH
# Description: Query MegaRAID information
#
#
# USAGE:
#   as a script:    %SystemRoot%\system32\WindowsPowerShell\v1.0\powershell.exe -NoLogo -NoProfile -ExecutionPolicy Bypass -File "C:\bin\zabbix\megaraid\megaraid.ps1" "<ITEM_TO_QUERY>" "ARG1"
#   as an item:     megaraid[<ITEM_TO_QUERY>,<ARG1>]
#

$version = "1.0.1"

$TW_CLI = "${env:ProgramFiles(x86)}\MegaRAID Storage Manager\storcli64.exe"

$ITEM = [string]$args[0]
$ARG1 = [string]$args[1]




function Call-Cli {
	Param(
		[Parameter(Mandatory=$True,Position=1)]
		[string]$ARG
	)
	$return=& $TW_CLI $ARG | Out-String
	return $return.Trim()
}


function Get-Controller {
	Param(
		[Parameter(Mandatory=$False,Position=1)]
		[string]$Ctl=""
	)
	$ret=Call-Cli "show J" | ConvertFrom-Json
	return ($ret.Controllers."Response Data")
}




# Query Information
switch($ITEM) {
	
	
	"numofctl" {
		$controllers=Get-Controller $ARG1
		Write-Host ($controllers."Number of Controllers")
	}
	
	
	
	
	"controller_discovery" {
		$controllers=Get-Controller $ARG1
		$count = ($controllers."System Overview" | Measure-Object).count
		$output =  "{`"data`":["
		foreach($controller in $controllers."System Overview") {
			$output+= "{"
			$output+= "`"{#CTL}`":`"/c$($controller.Ctl)`","
			$output+= "`"{#MODEL}`":`"$($controller.Model)`","
			$output+= "`"{#PORTS}`":`"$($controller.Ports)`","
			$output+= "`"{#PD}`":`"$($controller.PDs)`","
			$output+= "`"{#DG}`":`"$($controller.DGs)`","
			$output+= "`"{#VD}`":`"$($controller.VDs)`""
			$output+= "}"
			if ($count -gt 1) { $output+= "," }
			$count--
		}
		$output = $output + "]}"
		Write-Host $output
	}
	
	
	"controller_info" {
		$ret=Call-Cli "$ARG1 show J" | ConvertFrom-Json
		Write-Host ($ret.Controllers."Response Data" | ConvertTo-Json)
	}
	
	
	
	"vd_discovery" {
		$controllers=Get-Controller
		$o=@{"data" = @()}
		foreach($controller in $controllers."System Overview") {
			$Ctl="/c$($controller.Ctl)"
			$ret=Call-Cli "$Ctl show J" | ConvertFrom-Json
			$VDs=$ret.Controllers."Response Data"."VD LIST"
			foreach($VD in $VDs) {
				$VDNum=($VD."DG/VD").Split("/")
				$oVD=@{
					"{#VD}"     = "$Ctl/v$($VDNum[1])";
					"{#DG}"     = "$Ctl/d$($VDNum[0])";
					"{#CTL}"    = "$Ctl";
					"{#TYPE}"   = "$($VD.TYPE)";
					"{#STATE}"  = "$($VD.State)";
					"{#ACCESS}" = "$($VD.Access)";
					"{#NAME}"   = "$($VD.Name)"
				}
				$o.Data+=$oVD
			}
		}
		Write-Host ($o | ConvertTo-Json)
	}
	
	
	"vd_info" {
		if($ARG1 -eq "") {
			Write-Host "ZBX_NOTSUPPORTED"
			return
		}
		$CtlNum=($ARG1).Split("/")
		$ret=Call-Cli "$ARG1 show all J" | ConvertFrom-Json
		$VD=[pscustomobject]@{}
		$ret.Controllers."Response Data"."$ARG1" | Get-Member -MemberType NoteProperty | ForEach { $VD | Add-Member -Name "$($_.Name)" -Value $ret.Controllers."Response Data"."$ARG1"."$($_.Name)" -MemberType NoteProperty }
		$VDNum=($VD."DG/VD").Split("/")
		$VD | Add-Member -Name "VD" -Value "/$($CtlNum[1])/v$($VDNum[1])" -MemberType NoteProperty
		$VD | Add-Member -Name "DG" -Value "/$($CtlNum[1])/d$($VDNum[0])" -MemberType NoteProperty
		$ret.Controllers."Response Data"."VD$($VDNum[1]) Properties" | Get-Member -MemberType NoteProperty | ForEach { $VD | Add-Member -Name "$($_.Name)" -Value $ret.Controllers."Response Data"."VD$($VDNum[1]) Properties"."$($_.Name)" -MemberType NoteProperty }
		#$VD | Add-Member -Name "Number of Bytes" -Value (4096*$($VD."Number of Blocks")) -MemberType NoteProperty
		Write-Host ($VD | ConvertTo-Json)
	}
	
	
	
	"pd_discovery" {
		$controllers=Get-Controller
		$o=@{"data" = @()}
		foreach($controller in $controllers."System Overview") {
			$Ctl="/c$($controller.Ctl)"
			$ret=Call-Cli "$Ctl show J" | ConvertFrom-Json
			$PDs=$ret.Controllers."Response Data"."PD LIST"
			foreach($PD in $PDs) {
				$PDNum=($PD."EID:Slt").Split(":")
				$oPD=@{
					"{#PD}"    = "$Ctl/e$($PDNum[0])/s$($PDNum[1])";
					"{#CTL}"   = $CTL;
					"{#DG}"    = "$Ctl/d$($PD.DG)";
					"{#EID}"   = "$Ctl/e$($PDNum[0])";
					"{#STATE}" = $($PD.STate);
					"{#MODEL}" = $($PD.Model).Trim()
				}
				$o.Data+=$oPD
			}
		}
		Write-Host ($o | ConvertTo-Json)
	}
	
	
	"pd_info" {
		if($ARG1 -eq "") {
			Write-Host "ZBX_NOTSUPPORTED"
			return
		}
		$CtlNum=($ARG1).Split("/")
		$ret=Call-Cli "$ARG1 show all J" | ConvertFrom-Json
		$PD=[pscustomobject]@{}
		
		$ret1=$ret.Controllers."Response Data"."Drive $ARG1"
		$ret1 | Get-Member -MemberType NoteProperty | ForEach { $PD | Add-Member -Name "$($_.Name)" -Value $ret1."$($_.Name)" -MemberType NoteProperty }
		$PD.DG="/$($CtlNum[1])/d$($PD.DG)"
		$PDNum=($PD."EID:Slt").Split(":")
		$PD | Add-Member -Name "PD" -Value "/$($CtlNum[1])/e$($PDNum[0])/s$($PDNum[1])" -MemberType NoteProperty
		$PD | Add-Member -Name "EID" -Value "/$($CtlNum[1])/e$($PDNum[0])" -MemberType NoteProperty
		
		$ret1=$ret.Controllers."Response Data"."Drive $ARG1 - Detailed Information"."Drive $ARG1 State"
		$ret1 | Get-Member -MemberType NoteProperty | ForEach { $PD | Add-Member -Name "$($_.Name)" -Value $ret1."$($_.Name)" -MemberType NoteProperty }
		try { $PD | Add-Member -Name "Drive Temperature [C]" -Value $($PD."Drive Temperature" | Select-String -Pattern '([0-9.]+)C').Matches[0].Groups.Groups[1].Value -MemberType NoteProperty }
		catch { $PD | Add-Member -Name "Drive Temperature [C]" -Value -1 -MemberType NoteProperty }
		
		$ret1=$ret.Controllers."Response Data"."Drive $ARG1 - Detailed Information"."Drive $ARG1 Device attributes"
		$ret1 | Get-Member -MemberType NoteProperty | ForEach { $PD | Add-Member -Name "$($_.Name)" -Value $ret1."$($_.Name)" -MemberType NoteProperty }
		$PD | Add-Member -Name "Mfg Model" -Value "$($PD."Manufacturer Id".Trim()) $($PD."Model Number".Trim())" -MemberType NoteProperty
		try { $PD | Add-Member -Name "Size [sectors]" -Value $([convert]::toint64($($PD."Coerced size" | Select-String -Pattern '\[(0x[0-9a-f]+) Sectors\]').Matches[0].Groups.Groups[1].Value, 16)) -MemberType NoteProperty }
		catch { $PD | Add-Member -Name "Size [sectors]" -Value -1 -MemberType NoteProperty }
		
		try { $PD | Add-Member -Name "Sectorsize [B]" -Value (1 * $($PD."SeSz" | Select-String -Pattern '(\d+)\s*B').Matches[0].Groups.Groups[1].Value) -MemberType NoteProperty }
		catch {
			try { $PD | Add-Member -Name "Sectorsize [B]" -Value (1024 * $($PD."SeSz" | Select-String -Pattern '(\d+)\s*KB').Matches[0].Groups.Groups[1].Value) -MemberType NoteProperty }
			catch { $PD | Add-Member -Name "Size [sectors]" -Value -1 -MemberType NoteProperty }
		}

		$PD | Add-Member -Name "Size [B]" -Value ($($PD."Sectorsize [B]") * $($PD."Size [sectors]")) -MemberType NoteProperty

		
		$ret1=$ret.Controllers."Response Data"."Drive $ARG1 - Detailed Information"."Drive $ARG1 Policies/Settings"
		$ret1 | Get-Member -MemberType NoteProperty | ForEach { $PD | Add-Member -Name "$($_.Name)" -Value $ret1."$($_.Name)" -MemberType NoteProperty }
		
		#Write-Host ($ret.Controllers."Response Data" | ConvertTo-Json)
		Write-Host ($PD | ConvertTo-Json)
	}
	
	
}




